package com.shiguangjianke.smartfoodsystem.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.shiguangjianke.smartfoodsystem.entity.Dish;
import com.shiguangjianke.smartfoodsystem.entity.Order;
import com.shiguangjianke.smartfoodsystem.entity.Store;
import com.shiguangjianke.smartfoodsystem.entity.User;
import com.shiguangjianke.smartfoodsystem.mapper.OrderMapper;
import com.shiguangjianke.smartfoodsystem.service.DishService;
import com.shiguangjianke.smartfoodsystem.service.OrderService;
import com.shiguangjianke.smartfoodsystem.service.StoreService;
import com.shiguangjianke.smartfoodsystem.service.UserService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.lang.reflect.Field;
import java.util.*;
import java.util.Date;
import java.util.stream.Collectors;

/**
 * 订单Service实现类
 *
 * @author shiguangjianke
 * @since 2024-07-28
 */
@Service
public class OrderServiceImpl extends ServiceImpl<OrderMapper, Order> implements OrderService {

    @Autowired
    private StoreService storeService;
    
    @Autowired
    private DishService dishService;
    
    @Autowired
    private UserService userService;
    
    @Override
    public List<Order> getOrdersByMerchantId(Long merchantId) {
        // 1. 根据商家ID获取所有店面
        List<Store> stores = storeService.getByMerchantId(merchantId);
        if (stores == null || stores.isEmpty()) {
            return new ArrayList<>();
        }
        
        // 2. 获取所有店面ID
        List<Long> storeIds = stores.stream().map(Store::getId).collect(Collectors.toList());
        
        // 3. 根据店面ID获取所有菜品
        QueryWrapper<Dish> dishQueryWrapper = new QueryWrapper<>();
        dishQueryWrapper.in("store_id", storeIds);
        List<Dish> dishes = dishService.list(dishQueryWrapper);
        
        if (dishes == null || dishes.isEmpty()) {
            return new ArrayList<>();
        }
        
        // 4. 获取所有菜品ID
        List<Long> dishIds = dishes.stream().map(Dish::getId).collect(Collectors.toList());
        
        // 5. 根据菜品ID获取所有订单
        QueryWrapper<Order> orderQueryWrapper = new QueryWrapper<>();
        orderQueryWrapper.in("dish_id", dishIds);
        orderQueryWrapper.orderByDesc("create_time");
        
        List<Order> orders = this.list(orderQueryWrapper);
        
        // 6. 为订单添加菜品名称
        addDishNameToOrders(orders);
        
        return orders;
    }
    
    @Override
    public IPage<Order> getOrdersByMerchantId(Long merchantId, Integer page, Integer pageSize) {
        // 1. 根据商家ID获取所有店面
        List<Store> stores = storeService.getByMerchantId(merchantId);
        if (stores == null || stores.isEmpty()) {
            return new Page<>(page, pageSize);
        }
        
        // 2. 获取所有店面ID
        List<Long> storeIds = stores.stream().map(Store::getId).collect(Collectors.toList());
        
        // 3. 根据店面ID获取所有菜品
        QueryWrapper<Dish> dishQueryWrapper = new QueryWrapper<>();
        dishQueryWrapper.in("store_id", storeIds);
        List<Dish> dishes = dishService.list(dishQueryWrapper);
        
        if (dishes == null || dishes.isEmpty()) {
            return new Page<>(page, pageSize);
        }
        
        // 4. 获取所有菜品ID
        List<Long> dishIds = dishes.stream().map(Dish::getId).collect(Collectors.toList());
        
        // 5. 根据菜品ID获取分页订单
        QueryWrapper<Order> orderQueryWrapper = new QueryWrapper<>();
        orderQueryWrapper.in("dish_id", dishIds);
        orderQueryWrapper.orderByDesc("create_time");
        
        Page<Order> orderPage = new Page<>(page, pageSize);
        IPage<Order> resultPage = this.page(orderPage, orderQueryWrapper);
        
        // 6. 为订单添加菜品名称
        addDishNameToOrders(resultPage.getRecords());
        
        return resultPage;
    }
    
    /**
     * 为订单添加菜品名称和用户信息
     * @param orders 订单列表
     */
    private void addDishNameToOrders(List<Order> orders) {
        if (orders == null || orders.isEmpty()) {
            return;
        }
        
        // 获取所有订单的菜品ID
        List<Long> dishIds = orders.stream()
                .map(Order::getDishId)
                .distinct()
                .collect(Collectors.toList());
        
        // 批量查询菜品
        List<Dish> dishes = dishService.listByIds(dishIds);
        
        // 构建菜品ID到名称的映射
        Map<Long, String> dishNameMap = dishes.stream()
                .collect(Collectors.toMap(Dish::getId, Dish::getName));
        
        // 获取所有订单的用户ID
        List<Long> userIds = orders.stream()
                .map(Order::getUserId)
                .distinct()
                .collect(Collectors.toList());
        
        // 批量查询用户
        List<User> users = userService.listByIds(userIds);
        
        // 构建用户ID到用户信息的映射
        Map<Long, User> userMap = users.stream()
                .collect(Collectors.toMap(User::getId, user -> user));
        
        // 为每个订单设置菜品名称和用户信息
        for (Order order : orders) {
            order.setDishName(dishNameMap.getOrDefault(order.getDishId(), "未知菜品"));
            User user = userMap.get(order.getUserId());
            if (user != null) {
                order.setUsername(user.getUsername());
                order.setPhone(user.getPhone());
            }
        }
    }
    
    @Override
    public boolean updateOrderStatus(Long orderId, String status) {
        // 根据订单ID查询订单
        Order order = this.getById(orderId);
        if (order == null) {
            return false;
        }
        
        // 更新订单状态和更新时间
        order.setStatus(status);
        order.setUpdateTime(new Date());
        return this.updateById(order);
    }
    
    @Override
    public IPage<Order> getOrdersByUserId(Long userId, Integer page, Integer pageSize) {
        // 根据用户ID查询订单列表
        QueryWrapper<Order> orderQueryWrapper = new QueryWrapper<>();
        orderQueryWrapper.eq("user_id", userId);
        orderQueryWrapper.orderByDesc("create_time");
        
        Page<Order> orderPage = new Page<>(page, pageSize);
        IPage<Order> resultPage = this.page(orderPage, orderQueryWrapper);
        
        // 为订单添加菜品名称
        addDishNameToOrders(resultPage.getRecords());
        
        return resultPage;
    }
}
