package com.shiguangjianke.smartfoodsystem.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.shiguangjianke.smartfoodsystem.entity.Evaluation;
import com.shiguangjianke.smartfoodsystem.mapper.EvaluationMapper;
import com.shiguangjianke.smartfoodsystem.service.EvaluationService;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;

/**
 * 评价Service实现类
 *
 * @author shiguangjianke
 * @since 2024-07-28
 */
@Service
public class EvaluationServiceImpl extends ServiceImpl<EvaluationMapper, Evaluation> implements EvaluationService {
    
    @Override
    public List<Evaluation> getByStoreId(Long storeId) {
        return this.list(new QueryWrapper<Evaluation>().eq("store_id", storeId));
    }
    
    @Override
    public List<Evaluation> getByUserId(Long userId) {
        return this.list(new QueryWrapper<Evaluation>().eq("user_id", userId));
    }
    
    @Override
    public IPage<Evaluation> getByUserId(Long userId, int page, int pageSize) {
        IPage<Evaluation> evaluationPage = new Page<>(page, pageSize);
        return this.page(evaluationPage, new QueryWrapper<Evaluation>().eq("user_id", userId).orderByDesc("create_time"));
    }
    
    @Override
    public Double calculateAverageScore(Long storeId) {
        // 获取该店面的所有评价
        List<Evaluation> evaluations = this.getByStoreId(storeId);
        if (evaluations == null || evaluations.isEmpty()) {
            return 0.0;
        }
        
        // 计算总评分
        double totalScore = 0;
        for (Evaluation evaluation : evaluations) {
            totalScore += evaluation.getTasteScore() + evaluation.getEnvironmentScore() + 
                         evaluation.getServiceScore() + evaluation.getHygieneScore();
        }
        
        // 每个评价有4个维度，所以总维度数为评价数量*4
        return totalScore / (evaluations.size() * 4);
    }
    
    @Override
    public Integer getEvaluationCount(Long storeId) {
        return Math.toIntExact(this.count(new QueryWrapper<Evaluation>().eq("store_id", storeId)));
    }
    
    @Override
    public IPage<Evaluation> getByStoreId(Long storeId, int page, int pageSize) {
        IPage<Evaluation> evaluationPage = new Page<>(page, pageSize);
        return this.page(evaluationPage, new QueryWrapper<Evaluation>()
                .eq("store_id", storeId)
                .orderByDesc("create_time"));
    }
    
    @Override
    public boolean updateReply(Evaluation evaluation) {
        evaluation.setReplyTime(new Date());
        return this.updateById(evaluation);
    }
    
    @Override
    public IPage<Evaluation> getAllEvaluations(int page, int pageSize) {
        IPage<Evaluation> evaluationPage = new Page<>(page, pageSize);
        return this.page(evaluationPage, new QueryWrapper<Evaluation>()
                .orderByDesc("create_time"));
    }
    
    @Override
    public IPage<Evaluation> getEvaluationsByCondition(Integer status, Date startTime, Date endTime, int page, int pageSize) {
        QueryWrapper<Evaluation> queryWrapper = new QueryWrapper<>();
        
        // 根据评价状态筛选
        if (status != null) {
            queryWrapper.eq("status", status);
        }
        
        // 根据时间范围筛选
        if (startTime != null) {
            queryWrapper.ge("create_time", startTime);
        }
        if (endTime != null) {
            queryWrapper.le("create_time", endTime);
        }
        
        // 排序
        queryWrapper.orderByDesc("create_time");
        
        IPage<Evaluation> evaluationPage = new Page<>(page, pageSize);
        return this.page(evaluationPage, queryWrapper);
    }
    
    @Override
    public boolean updateEvaluationStatus(Long id, Integer status) {
        Evaluation evaluation = new Evaluation();
        evaluation.setId(id);
        evaluation.setStatus(status);
        evaluation.setUpdateTime(new Date());
        return this.updateById(evaluation);
    }
}
