package com.shiguangjianke.smartfoodsystem.controller;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.shiguangjianke.smartfoodsystem.entity.User;
import com.shiguangjianke.smartfoodsystem.service.UserService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

/**
 * 用户Controller
 *
 * @author shiguangjianke
 * @since 2024-07-28
 */
@RestController
@RequestMapping("/api/user")
public class UserController extends BaseController {

    @Autowired
    private UserService userService;

    /**
     * 获取商家列表，支持分页和搜索
     *
     * @param keyword 搜索关键词
     * @param merchantApplyStatus 商家申请状态
     * @param page 页码
     * @param pageSize 每页数量
     * @return 商家列表
     */
    @GetMapping("/merchant/list")
    public Result getMerchantList(
            @RequestParam(required = false) String keyword,
            @RequestParam(required = false) Integer merchantApplyStatus,
            @RequestParam(defaultValue = "1") Integer page,
            @RequestParam(defaultValue = "10") Integer pageSize) {
        QueryWrapper<User> queryWrapper = new QueryWrapper<>();

        // 只查询商家用户
        queryWrapper.eq("user_type", 2);

        // 搜索关键词
        if (keyword != null && !keyword.isEmpty()) {
            queryWrapper.like("username", keyword).or().like("phone", keyword);
        }

        // 商家申请状态
        if (merchantApplyStatus != null) {
            queryWrapper.eq("merchant_apply_status", merchantApplyStatus);
        }

        // 按创建时间降序
        queryWrapper.orderByDesc("create_time");

        // 分页查询
        Page<User> userPage = new Page<>(page, pageSize);
        IPage<User> result = userService.page(userPage, queryWrapper);

        return success(result);
    }

    /**
     * 根据ID获取商家详情
     *
     * @param id 商家ID
     * @return 商家详情
     */
    @GetMapping("/merchant/detail/{id}")
    public Result getMerchantDetail(@PathVariable Long id) {
        User user = userService.getById(id);
        if (user == null || user.getUserType() != 2) {
            return error(404, "商家不存在");
        }
        return success(user);
    }

    /**
     * 更新商家入驻申请状态
     *
     * @param id 商家ID
     * @param status 申请状态
     * @return 更新结果
     */
    @PutMapping("/merchant/apply/status")
    public Result updateMerchantApplyStatus(@RequestParam Long id, @RequestParam Integer status) {
        boolean result = userService.updateMerchantApplyStatus(id, status);
        if (result) {
            return success("更新成功");
        } else {
            return error("更新失败");
        }
    }

    /**
     * 删除商家
     *
     * @param id 商家ID
     * @return 删除结果
     */
    @DeleteMapping("/merchant/{id}")
    public Result deleteMerchant(@PathVariable Long id) {
        User user = userService.getById(id);
        if (user == null || user.getUserType() != 2) {
            return error(404, "商家不存在");
        }
        boolean result = userService.removeById(id);
        if (result) {
            return success("删除成功");
        } else {
            return error("删除失败");
        }
    }

    /**
     * 更新商家信息
     *
     * @param user 商家信息
     * @return 更新结果
     */
    @PutMapping("/merchant/update")
    public Result updateMerchant(@RequestBody User user) {
        // 确保只能更新商家信息
        user.setUserType(2);
        boolean result = userService.updateById(user);
        if (result) {
            return success("更新成功");
        } else {
            return error("更新失败");
        }
    }
}
