package com.shiguangjianke.smartfoodsystem.controller;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.shiguangjianke.smartfoodsystem.entity.Store;
import com.shiguangjianke.smartfoodsystem.service.StoreService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 店面Controller
 *
 * @author shiguangjianke
 * @since 2024-07-28
 */
@RestController
@RequestMapping("/api/store")
public class StoreController extends BaseController {
    
    @Autowired
    private StoreService storeService;
    
    /**
     * 首页推荐
     *
     * @return 推荐店面列表
     */
    @GetMapping("/recommend")
    public Result recommendStores() {
        // 首页推荐逻辑：按评分排序，返回前6个营业中的店面
        QueryWrapper<Store> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("status", 1) // 只显示营业中的店面
                .orderByDesc("average_score") // 按评分降序
                .last("LIMIT 6");
        return success(storeService.list(queryWrapper));
    }

    /**
     * 获取所有店面列表，支持分页和搜索
     *
     * @param keyword 搜索关键词
     * @param status 店面状态
     * @param page 页码
     * @param pageSize 每页数量
     * @return 店面列表
     */
    @GetMapping("/list")
    public Result getStoreList(
            @RequestParam(required = false) String keyword,
            @RequestParam(required = false) Integer status,
            @RequestParam(defaultValue = "1") Integer page,
            @RequestParam(defaultValue = "10") Integer pageSize) {
        QueryWrapper<Store> queryWrapper = new QueryWrapper<>();
        
        // 只显示营业中的店面
        queryWrapper.eq("status", 1);
        
        // 搜索关键词
        if (keyword != null && !keyword.isEmpty()) {
            queryWrapper.like("name", keyword).or().like("description", keyword);
        }
        
        // 按评分降序
        queryWrapper.orderByDesc("average_score");
        
        // 分页查询
        Page<Store> storePage = new Page<>(page, pageSize);
        IPage<Store> result = storeService.page(storePage, queryWrapper);
        
        return success(result);
    }
    
    /**
     * 根据ID获取店面详情
     *
     * @param id 店面ID
     * @return 店面详情
     */
    @GetMapping("/detail/{id}")
    public Result getStoreDetail(@PathVariable Long id) {
        Store store = storeService.getById(id);
        if (store == null) {
            return error(404, "店面不存在");
        }
        return success(store);
    }
    
    /**
     * 根据商家ID获取店面列表
     *
     * @param merchantId 商家ID
     * @return 店面列表
     */
    @GetMapping("/list/merchant/{merchantId}")
    public Result getStoreListByMerchantId(@PathVariable Long merchantId) {
        List<Store> stores = storeService.getByMerchantId(merchantId);
        return success(stores);
    }
    
    /**
     * 添加店面
     *
     * @param store 店面信息
     * @return 添加结果
     */
    @PostMapping("/add")
    public Result addStore(@RequestBody Store store) {
        store.setStatus(0); // 待审核
        storeService.save(store);
        return success("添加成功，等待审核");
    }
    
    /**
     * 更新店面信息
     *
     * @param store 店面信息
     * @return 更新结果
     */
    @PutMapping("/update")
    public Result updateStore(@RequestBody Store store) {
        storeService.updateById(store);
        return success("更新成功");
    }
    
    /**
     * 删除店面
     *
     * @param id 店面ID
     * @return 删除结果
     */
    @DeleteMapping("/delete/{id}")
    public Result deleteStore(@PathVariable Long id) {
        storeService.removeById(id);
        return success("删除成功");
    }
    
    /**
     * 根据状态获取店面列表
     *
     * @param status 店面状态
     * @return 店面列表
     */
    @GetMapping("/list/status/{status}")
    public Result getStoreListByStatus(@PathVariable Integer status) {
        List<Store> stores = storeService.getByStatus(status);
        return success(stores);
    }
    
    /**
     * 审核店面
     *
     * @param id 店面ID
     * @param status 审核结果状态 (1: 通过, 3: 拒绝)
     * @return 审核结果
     */
    @PutMapping("/audit/{id}")
    public Result auditStore(@PathVariable Long id, @RequestParam Integer status) {
        Store store = storeService.getById(id);
        if (store == null) {
            return error(404, "店面不存在");
        }
        
        // 审核通过：设置为营业状态(1)，审核拒绝：设置为拒绝状态(4)
        if (status == 1) {
            store.setStatus(1); // 通过并设置为营业中
        } else if (status == 3) {
            store.setStatus(4); // 拒绝
        } else {
            return error(400, "无效的审核状态");
        }
        
        storeService.updateById(store);
        return success("审核成功");
    }
    
    /**
     * 更新店面营业状态
     *
     * @param id 店面ID
     * @param status 营业状态 (1: 营业中, 2: 休息中, 3: 已关闭)
     * @return 更新结果
     */
    @PutMapping("/status/{id}")
    public Result updateStoreStatus(@PathVariable Long id, @RequestParam Integer status) {
        Store store = storeService.getById(id);
        if (store == null) {
            return error(404, "店面不存在");
        }
        
        // 只有已审核通过的店铺才能修改营业状态
        if (store.getStatus() == 0 || store.getStatus() == 4) {
            return error(400, "该店铺未通过审核，无法修改营业状态");
        }
        
        // 验证营业状态值
        if (status != 1 && status != 2 && status != 3) {
            return error(400, "无效的营业状态");
        }
        
        store.setStatus(status);
        storeService.updateById(store);
        return success("营业状态更新成功");
    }
    
    /**
     * 获取所有店面列表（包括待审核），支持分页和搜索
     *
     * @param keyword 搜索关键词
     * @param status 店面状态
     * @param merchantId 商家ID
     * @param page 页码
     * @param pageSize 每页数量
     * @return 店面列表
     */
    @GetMapping("/admin/list")
    public Result getAdminStoreList(
            @RequestParam(required = false) String keyword,
            @RequestParam(required = false) Integer status,
            @RequestParam(required = false) Long merchantId,
            @RequestParam(defaultValue = "1") Integer page,
            @RequestParam(defaultValue = "10") Integer pageSize) {
        QueryWrapper<Store> queryWrapper = new QueryWrapper<>();
        
        // 搜索关键词
        if (keyword != null && !keyword.isEmpty()) {
            queryWrapper.like("name", keyword).or().like("description", keyword);
        }
        
        // 店面状态
        if (status != null) {
            queryWrapper.eq("status", status);
        }
        
        // 商家ID
        if (merchantId != null) {
            queryWrapper.eq("merchant_id", merchantId);
        }
        
        // 按创建时间降序
        queryWrapper.orderByDesc("create_time");
        
        // 分页查询
        Page<Store> storePage = new Page<>(page, pageSize);
        IPage<Store> result = storeService.page(storePage, queryWrapper);
        
        return success(result);
    }
}
