package com.shiguangjianke.smartfoodsystem.controller;

import com.shiguangjianke.smartfoodsystem.entity.Order;
import com.shiguangjianke.smartfoodsystem.service.OrderService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.Date;

/**
 * 订单Controller
 *
 * @author shiguangjianke
 * @since 2024-07-28
 */
@RestController
@RequestMapping("/api/order")
public class OrderController extends BaseController {

    @Autowired
    private OrderService orderService;

    /**
     * 创建订单
     *
     * @param order 订单信息
     * @return 创建结果
     */
    @PostMapping("/add")
    public Result addOrder(@RequestBody Order order) {
        // 设置默认状态为pending
        order.setStatus("pending");
        // 设置创建时间和更新时间
        Date now = new Date();
        order.setCreateTime(now);
        order.setUpdateTime(now);
        // 保存订单
        boolean result = orderService.save(order);
        if (result) {
            return success("订单创建成功");
        } else {
            return error(500, "订单创建失败");
        }
    }
    
    /**
     * 商家端获取订单消息
     *
     * @param merchantId 商家ID
     * @param page 页码
     * @param pageSize 每页数量
     * @return 订单列表
     */
    @GetMapping("/merchant/messages")
    public Result getMerchantOrderMessages(
            @RequestParam Long merchantId,
            @RequestParam(defaultValue = "1") Integer page,
            @RequestParam(defaultValue = "10") Integer pageSize) {
        return success(orderService.getOrdersByMerchantId(merchantId, page, pageSize));
    }
    
    /**
     * 商家端获取所有订单（不分页）
     *
     * @param merchantId 商家ID
     * @return 订单列表
     */
    @GetMapping("/merchant/all")
    public Result getMerchantAllOrders(@RequestParam Long merchantId) {
        return success(orderService.getOrdersByMerchantId(merchantId));
    }
    
    /**
     * 更新订单状态
     *
     * @param orderId 订单ID
     * @param status 订单状态
     * @return 更新结果
     */
    @PutMapping("/update/status")
    public Result updateOrderStatus(@RequestParam Long orderId, @RequestParam String status) {
        boolean result = orderService.updateOrderStatus(orderId, status);
        if (result) {
            return success("订单状态更新成功");
        } else {
            return error(500, "订单状态更新失败");
        }
    }
    
    /**
     * 用户端获取订单列表
     *
     * @param userId 用户ID
     * @param page 页码
     * @param pageSize 每页数量
     * @return 订单列表
     */
    @GetMapping("/user/list")
    public Result getUserOrderList(
            @RequestParam Long userId,
            @RequestParam(defaultValue = "1") Integer page,
            @RequestParam(defaultValue = "10") Integer pageSize) {
        return success(orderService.getOrdersByUserId(userId, page, pageSize));
    }
}
