package com.shiguangjianke.smartfoodsystem.controller;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.shiguangjianke.smartfoodsystem.entity.Feedback;
import com.shiguangjianke.smartfoodsystem.service.FeedbackService;
import com.shiguangjianke.smartfoodsystem.service.UserService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.Date;
import java.util.List;
/**
 * 反馈Controller
 *
 * @author shiguangjianke
 * @since 2024-07-28
 */
@RestController
@RequestMapping("/api/feedback")
public class FeedbackController extends BaseController {
    
    @Autowired
    private FeedbackService feedbackService;
    
    /**
     * 获取所有反馈列表
     *
     * @return 反馈列表
     */
    @GetMapping("/list")
    public Result getFeedbackList() {
        List<Feedback> feedbacks = feedbackService.list();
        return success(feedbacks);
    }
    
    /**
     * 管理员获取反馈列表，支持分页和搜索
     *
     * @param keyword 搜索关键词
     * @param status 反馈状态
     * @param type 反馈类型
     * @param page 页码
     * @param pageSize 每页数量
     * @return 反馈列表
     */
    @GetMapping("/admin/list")
    public Result getAdminFeedbackList(
            @RequestParam(required = false) String keyword,
            @RequestParam(required = false) Integer status,
            @RequestParam(required = false) Integer type,
            @RequestParam(defaultValue = "1") Integer page,
            @RequestParam(defaultValue = "10") Integer pageSize) {
        QueryWrapper<Feedback> queryWrapper = new QueryWrapper<>();
        
        // 搜索关键词
        if (keyword != null && !keyword.isEmpty()) {
            queryWrapper.like("content", keyword);
        }
        
        // 反馈状态
        if (status != null) {
            queryWrapper.eq("status", status);
        }
        
        // 反馈类型
        if (type != null) {
            queryWrapper.eq("type", type);
        }
        
        // 按创建时间降序
        queryWrapper.orderByDesc("create_time");
        
        // 分页查询
        Page<Feedback> feedbackPage = new Page<>(page, pageSize);
        IPage<Feedback> result = feedbackService.page(feedbackPage, queryWrapper);
        
        return success(result);
    }
    
    /**
     * 根据用户ID获取反馈列表
     *
     * @param userId 用户ID
     * @param page 页码
     * @param pageSize 每页大小
     * @return 反馈列表
     */
    @GetMapping("/list/user/{userId}")
    public Result getFeedbackListByUserId(@PathVariable Long userId, 
                                        @RequestParam(defaultValue = "1") Integer page, 
                                        @RequestParam(defaultValue = "10") Integer pageSize) {
        IPage<Feedback> feedbackPage = feedbackService.getByUserId(userId, page, pageSize);
        return success(feedbackPage);
    }
    
    /**
     * 根据状态获取反馈列表
     *
     * @param status 反馈状态
     * @return 反馈列表
     */
    @GetMapping("/list/status/{status}")
    public Result getFeedbackListByStatus(@PathVariable Integer status) {
        List<Feedback> feedbacks = feedbackService.getByStatus(status);
        return success(feedbacks);
    }
    
    /**
     * 添加反馈
     *
     * @param feedback 反馈信息
     * @return 添加结果
     */
    @PostMapping("/add")
    public Result addFeedback(@RequestBody Feedback feedback) {
        feedback.setStatus(0); // 待处理
        feedbackService.save(feedback);
        return success("反馈成功");
    }
    
    /**
     * 更新反馈状态
     *
     * @param id 反馈ID
     * @param status 反馈状态
     * @return 更新结果
     */
    @PutMapping("/update/status/{id}")
    public Result updateFeedbackStatus(@PathVariable Long id, @RequestParam Integer status) {
        feedbackService.updateStatus(id, status);
        return success("状态更新成功");
    }
    
    /**
     * 处理反馈
     *
     * @param id 反馈ID
     * @param feedback 反馈信息（包含处理结果和状态）
     * @return 更新结果
     */
    @PutMapping("/admin/handle/{id}")
    public Result handleFeedback(@PathVariable Long id, @RequestBody Feedback feedback) {
        try {
            System.out.println("处理反馈请求，ID: " + id);
            System.out.println("接收到的反馈信息: " + feedback);
            
            Feedback existingFeedback = new Feedback();
            existingFeedback.setId(id);
            existingFeedback.setStatus(feedback.getStatus());
            existingFeedback.setHandleResult(feedback.getHandleResult());
            // 暂时不设置handleAdminId，设为null避免外键约束问题
            existingFeedback.setHandleAdminId(null);
            existingFeedback.setHandleTime(new Date());
            
            System.out.println("准备更新的反馈信息: " + existingFeedback);
            boolean result = feedbackService.updateById(existingFeedback);
            System.out.println("更新结果: " + result);
            
            return success("反馈处理成功");
        } catch (Exception e) {
            System.out.println("处理反馈失败，错误信息: " + e.getMessage());
            e.printStackTrace();
            return error("处理反馈失败");
        }
    }
}
