package com.shiguangjianke.smartfoodsystem.controller;

import com.shiguangjianke.smartfoodsystem.entity.Favorite;
import com.shiguangjianke.smartfoodsystem.service.FavoriteService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 收藏Controller
 *
 * @author shiguangjianke
 * @since 2024-07-28
 */
@RestController
@RequestMapping("/api/favorite")
public class FavoriteController extends BaseController {
    
    @Autowired
    private FavoriteService favoriteService;
    
    /**
     * 根据用户ID获取收藏列表
     *
     * @param userId 用户ID
     * @return 收藏列表
     */
    @GetMapping("/list/{userId}")
    public Result getFavoriteList(@PathVariable Long userId) {
        return success(favoriteService.getFavoriteStores(userId));
    }
    
    /**
     * 添加收藏
     *
     * @param favorite 收藏信息
     * @return 添加结果
     */
    @PostMapping("/add")
    public Result addFavorite(@RequestBody Favorite favorite) {
        // 检查是否已经收藏
        boolean isFavorite = favoriteService.isFavorite(favorite.getUserId(), favorite.getStoreId());
        if (isFavorite) {
            return error(400, "已经收藏过该店面");
        }
        
        favoriteService.save(favorite);
        return success("收藏成功");
    }
    
    /**
     * 取消收藏
     *
     * @param userId 用户ID
     * @param storeId 店面ID
     * @return 取消收藏结果
     */
    @DeleteMapping("/cancel/{userId}/{storeId}")
    public Result cancelFavorite(@PathVariable Long userId, @PathVariable Long storeId) {
        Favorite favorite = favoriteService.getByUserIdAndStoreId(userId, storeId);
        if (favorite == null) {
            return error(404, "未收藏该店面");
        }
        
        favoriteService.removeById(favorite.getId());
        return success("取消收藏成功");
    }
    
    /**
     * 判断是否收藏
     *
     * @param userId 用户ID
     * @param storeId 店面ID
     * @return 判断结果
     */
    @GetMapping("/check/{userId}/{storeId}")
    public Result checkFavorite(@PathVariable Long userId, @PathVariable Long storeId) {
        boolean isFavorite = favoriteService.isFavorite(userId, storeId);
        return success(isFavorite);
    }
}
