package com.shiguangjianke.smartfoodsystem.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.shiguangjianke.smartfoodsystem.entity.Evaluation;
import com.shiguangjianke.smartfoodsystem.entity.User;
import com.shiguangjianke.smartfoodsystem.service.EvaluationService;
import com.shiguangjianke.smartfoodsystem.service.StoreService;
import com.shiguangjianke.smartfoodsystem.service.UserService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.Date;
import java.util.List;

/**
 * 评价Controller
 *
 * @author shiguangjianke
 * @since 2024-07-28
 */
@RestController
@RequestMapping("/api/evaluation")
public class EvaluationController extends BaseController {
    
    @Autowired
    private EvaluationService evaluationService;
    
    @Autowired
    private StoreService storeService;
    
    @Autowired
    private UserService userService;
    
    /**
     * 根据店面ID获取评价列表
     *
     * @param storeId 店面ID
     * @return 评价列表
     */
    @GetMapping("/list/{storeId}")
    public Result getEvaluationListByStoreId(@PathVariable Long storeId) {
        List<Evaluation> evaluations = evaluationService.getByStoreId(storeId);
        return success(evaluations);
    }
    
    /**
     * 根据用户ID获取评价列表
     *
     * @param userId 用户ID
     * @param page 页码
     * @param pageSize 每页大小
     * @return 评价列表
     */
    @GetMapping("/list/user/{userId}")
    public Result getEvaluationListByUserId(@PathVariable Long userId, 
                                          @RequestParam(defaultValue = "1") Integer page, 
                                          @RequestParam(defaultValue = "10") Integer pageSize) {
        IPage<Evaluation> evaluationPage = evaluationService.getByUserId(userId, page, pageSize);
        return success(evaluationPage);
    }
    
    /**
     * 添加评价
     *
     * @param evaluation 评价信息
     * @return 添加结果
     */
    @PostMapping("/add")
    public Result addEvaluation(@RequestBody Evaluation evaluation) {
        // 调试：输出接收到的评价数据
        System.out.println("接收到的评价数据: " + evaluation);
        System.out.println("用户ID: " + evaluation.getUserId());
        
        // 1. 验证用户ID是否存在
        Long userId = evaluation.getUserId();
        System.out.println("获取到的用户ID: " + userId);
        
        if (userId == null) {
            System.out.println("用户ID为空，返回400错误");
            return error(400, "用户ID不能为空");
        }
        
        // 2. 检查用户是否存在
        User user = userService.getById(userId);
        System.out.println("查询用户结果: " + user);
        
        if (user == null) {
            System.out.println("用户不存在，返回404错误");
            return error(404, "用户不存在");
        }
        
        // 保存评价
        evaluationService.save(evaluation);
        
        // 更新店面的平均评分和评价数量
        Long storeId = evaluation.getStoreId();
        Double averageScore = evaluationService.calculateAverageScore(storeId);
        Integer evaluationCount = evaluationService.getEvaluationCount(storeId);
        storeService.updateAverageScore(storeId, averageScore);
        storeService.updateEvaluationCount(storeId, evaluationCount);
        
        return success("评价成功");
    }
    
    /**
     * 删除评价
     *
     * @param id 评价ID
     * @return 删除结果
     */
    @DeleteMapping("/delete/{id}")
    public Result deleteEvaluation(@PathVariable Long id) {
        // 获取评价信息，用于后续更新店面评分
        Evaluation evaluation = evaluationService.getById(id);
        if (evaluation == null) {
            return error(404, "评价不存在");
        }
        
        // 删除评价
        evaluationService.removeById(id);
        
        // 更新店面的平均评分和评价数量
        Long storeId = evaluation.getStoreId();
        Double averageScore = evaluationService.calculateAverageScore(storeId);
        Integer evaluationCount = evaluationService.getEvaluationCount(storeId);
        storeService.updateAverageScore(storeId, averageScore);
        storeService.updateEvaluationCount(storeId, evaluationCount);
        
        return success("删除成功");
    }
    
    /**
     * 分页查询所有评价（管理端）
     *
     * @param page 页码
     * @param pageSize 每页大小
     * @return 分页评价列表
     */
    @GetMapping("/admin/list")
    public Result getAllEvaluations(
            @RequestParam(defaultValue = "1") Integer page,
            @RequestParam(defaultValue = "10") Integer pageSize) {
        IPage<Evaluation> evaluationPage = evaluationService.getAllEvaluations(page, pageSize);
        return success(evaluationPage);
    }
    
    /**
     * 根据条件分页查询评价（管理端）
     *
     * @param status 评价状态
     * @param startTime 开始时间
     * @param endTime 结束时间
     * @param page 页码
     * @param pageSize 每页大小
     * @return 分页评价列表
     */
    @GetMapping("/admin/list/condition")
    public Result getEvaluationsByCondition(
            @RequestParam(required = false) Integer status,
            @RequestParam(required = false) Date startTime,
            @RequestParam(required = false) Date endTime,
            @RequestParam(defaultValue = "1") Integer page,
            @RequestParam(defaultValue = "10") Integer pageSize) {
        IPage<Evaluation> evaluationPage = evaluationService.getEvaluationsByCondition(
                status, startTime, endTime, page, pageSize);
        return success(evaluationPage);
    }
    
    /**
     * 更新评价状态（管理端）
     *
     * @param id 评价ID
     * @param status 评价状态
     * @return 更新结果
     */
    @PutMapping("/admin/status")
    public Result updateEvaluationStatus(
            @RequestParam Long id,
            @RequestParam Integer status) {
        boolean result = evaluationService.updateEvaluationStatus(id, status);
        if (result) {
            return success("更新成功");
        } else {
            return error(500, "更新失败");
        }
    }
    
    /**
     * 根据店面ID分页获取评价列表（商家端）
     *
     * @param storeId 店面ID
     * @param page 页码，默认为1
     * @param pageSize 每页大小，默认为10
     * @return 评价列表
     */
    @GetMapping("/list/store/{storeId}")
    public Result getEvaluationListByStoreId(@PathVariable Long storeId, 
                                            @RequestParam(defaultValue = "1") int page, 
                                            @RequestParam(defaultValue = "10") int pageSize) {
        IPage<Evaluation> evaluationPage = evaluationService.getByStoreId(storeId, page, pageSize);
        return success(evaluationPage);
    }
    
    /**
     * 更新评价回复
     *
     * @param evaluation 评价信息（包含回复内容）
     * @return 更新结果
     */
    @PutMapping("/reply/update")
    public Result updateEvaluationReply(@RequestBody Evaluation evaluation) {
        boolean result = evaluationService.updateReply(evaluation);
        if (result) {
            return success("回复更新成功");
        } else {
            return error(500, "回复更新失败");
        }
    }
}
