package com.shiguangjianke.smartfoodsystem.controller;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.shiguangjianke.smartfoodsystem.entity.Dish;
import com.shiguangjianke.smartfoodsystem.service.DishService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;

/**
 * 菜品Controller
 *
 * @author shiguangjianke
 * @since 2024-07-28
 */
@RestController
@RequestMapping("/api/dish")
public class DishController extends BaseController {
    
    @Autowired
    private DishService dishService;
    
    /**
     * 根据店面ID获取菜品列表
     *
     * @param storeId 店面ID
     * @return 菜品列表
     */
    @GetMapping("/list/{storeId}")
    public Result getDishListByStoreId(@PathVariable Long storeId) {
        QueryWrapper<Dish> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("store_id", storeId);
        
        List<Dish> dishes = dishService.list(queryWrapper);
        return success(dishes);
    }
    
    /**
     * 根据ID获取菜品详情
     *
     * @param id 菜品ID
     * @return 菜品详情
     */
    @GetMapping("/detail/{id}")
    public Result getDishDetail(@PathVariable Long id) {
        Dish dish = dishService.getById(id);
        if (dish == null) {
            return error(404, "菜品不存在");
        }
        return success(dish);
    }
    
    /**
     * 添加菜品
     *
     * @param dish 菜品信息
     * @return 添加结果
     */
    @PostMapping("/add")
    public Result addDish(@RequestBody Dish dish) {
        dishService.save(dish);
        return success("添加成功");
    }
    
    /**
     * 更新菜品信息
     *
     * @param dish 菜品信息
     * @return 更新结果
     */
    @PutMapping("/update")
    public Result updateDish(@RequestBody Dish dish) {
        dishService.updateById(dish);
        return success("更新成功");
    }
    
    /**
     * 删除菜品
     *
     * @param id 菜品ID
     * @return 删除结果
     */
    @DeleteMapping("/delete/{id}")
    public Result deleteDish(@PathVariable Long id) {
        dishService.removeById(id);
        return success("删除成功");
    }
    
    /**
     * 修改菜品状态
     *
     * @param id     菜品ID
     * @param status 菜品状态（on_sale：上架，off_sale：下架）
     * @return 修改结果
     */
    @PutMapping("/update/status/{id}")
    public Result updateDishStatus(@PathVariable Long id, @RequestBody Map<String, String> requestBody) {
        // 1. 验证参数
        if (requestBody == null || !requestBody.containsKey("status")) {
            return error(400, "参数错误");
        }
        String status = requestBody.get("status");
        if (!"on_sale".equals(status) && !"off_sale".equals(status)) {
            return error(400, "状态值无效");
        }
        // 2. 查询菜品
        Dish dish = dishService.getById(id);
        if (dish == null) {
            return error(404, "菜品不存在");
        }
        // 3. 更新状态
        dish.setStatus(status);
        dishService.updateById(dish);
        // 4. 返回结果
        return success("状态更新成功");
    }

}
