package com.shiguangjianke.smartfoodsystem.controller;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.shiguangjianke.smartfoodsystem.entity.Dish;
import com.shiguangjianke.smartfoodsystem.entity.Order;
import com.shiguangjianke.smartfoodsystem.entity.Store;
import com.shiguangjianke.smartfoodsystem.service.DishService;
import com.shiguangjianke.smartfoodsystem.service.OrderService;
import com.shiguangjianke.smartfoodsystem.service.StoreService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 数据统计Controller
 *
 * @author shiguangjianke
 * @since 2024-07-28
 */
@RestController
@RequestMapping("/api/data-statistics")
public class DataStatisticsController extends BaseController {

    @Autowired
    private OrderService orderService;

    @Autowired
    private StoreService storeService;

    @Autowired
    private DishService dishService;

    /**
     * 获取数据统计基本信息
     *
     * @param merchantId 商家ID
     * @return 基本统计数据
     */
    @GetMapping("/basic")
    public Result getBasicStatistics(@RequestParam Long merchantId) {
        // 获取商家所有店面
        List<Store> stores = storeService.getByMerchantId(merchantId);
        if (stores.isEmpty()) {
            return success(new HashMap<String, Object>() {
                {
                    put("totalOrders", 0);
                    put("totalRevenue", new BigDecimal(0));
                    put("averageOrderPrice", new BigDecimal(0));
                    put("customerCount", 0);
                }
            });
        }

        // 构建店面ID列表
        List<Long> storeIds = stores.stream().map(Store::getId).toList();

        // 获取商家所有菜品
        QueryWrapper<Dish> dishQueryWrapper = new QueryWrapper<>();
        dishQueryWrapper.in("store_id", storeIds);
        List<Dish> dishes = dishService.list(dishQueryWrapper);
        if (dishes.isEmpty()) {
            return success(new HashMap<String, Object>() {
                {
                    put("totalOrders", 0);
                    put("totalRevenue", new BigDecimal(0));
                    put("averageOrderPrice", new BigDecimal(0));
                    put("customerCount", 0);
                }
            });
        }

        // 构建菜品ID列表
        List<Long> dishIds = dishes.stream().map(Dish::getId).toList();

        // 查询所有订单
        QueryWrapper<Order> orderQueryWrapper = new QueryWrapper<>();
        orderQueryWrapper.in("dish_id", dishIds);
        List<Order> orders = orderService.list(orderQueryWrapper);

        // 统计总订单数
        int totalOrders = orders.size();

        // 统计总营业额
        BigDecimal totalRevenue = orders.stream()
                .map(Order::getTotalPrice)
                .reduce(BigDecimal.ZERO, BigDecimal::add);

        // 计算客单价
        BigDecimal averageOrderPrice = totalOrders > 0 ? totalRevenue.divide(new BigDecimal(totalOrders), 2, BigDecimal.ROUND_HALF_UP) : new BigDecimal(0);

        // 统计顾客数（去重）
        int customerCount = (int) orders.stream()
                .map(Order::getUserId)
                .distinct()
                .count();

        // 构建返回结果
        Map<String, Object> result = new HashMap<>();
        result.put("totalOrders", totalOrders);
        result.put("totalRevenue", totalRevenue);
        result.put("averageOrderPrice", averageOrderPrice);
        result.put("customerCount", customerCount);

        return success(result);
    }

    /**
     * 获取订单趋势数据
     *
     * @param merchantId 商家ID
     * @param timeRange 时间范围（week/month/quarter/year/7days/30days/90days/180days）
     * @return 订单趋势数据
     */
    @GetMapping("/order/trend")
    public Result getOrderTrend(
            @RequestParam Long merchantId,
            @RequestParam(defaultValue = "week") String timeRange) {
        try {
            // 从数据库获取真实订单趋势数据
            List<String> dates = new ArrayList<>();
            List<Integer> orderCounts = new ArrayList<>();
            
            // 计算时间范围
            Calendar calendar = Calendar.getInstance();
            Date endDate = calendar.getTime();
            Date startDate = null;
            int interval = Calendar.DAY_OF_MONTH; // 时间间隔，默认为1天
            
            // 处理前端传递的时间范围值
            switch (timeRange) {
                case "week":
                case "7days":
                    // 最近7天
                    calendar.add(Calendar.DAY_OF_MONTH, -6);
                    startDate = calendar.getTime();
                    break;
                case "month":
                case "30days":
                    // 最近30天
                    calendar.add(Calendar.DAY_OF_MONTH, -29);
                    startDate = calendar.getTime();
                    break;
                case "quarter":
                case "90days":
                    // 最近90天
                    calendar.add(Calendar.DAY_OF_MONTH, -89);
                    startDate = calendar.getTime();
                    break;
                case "180days":
                    // 最近180天
                    calendar.add(Calendar.DAY_OF_MONTH, -179);
                    startDate = calendar.getTime();
                    break;
                case "year":
                    // 最近12个月
                    calendar.add(Calendar.MONTH, -11);
                    startDate = calendar.getTime();
                    interval = Calendar.MONTH;
                    break;
                default:
                    // 默认最近7天
                    calendar.add(Calendar.DAY_OF_MONTH, -6);
                    startDate = calendar.getTime();
            }
            
            // 获取商家所有店面
            List<Store> stores = storeService.getByMerchantId(merchantId);
            if (stores == null || stores.isEmpty()) {
                // 如果没有店面，返回空数据
                Map<String, Object> result = new HashMap<>();
                result.put("dates", dates);
                result.put("orderCounts", orderCounts);
                return success(result);
            }
            
            // 构建店面ID列表
            List<Long> storeIds = stores.stream().map(Store::getId).toList();
            
            // 查询该商家所有店面的所有菜品
            QueryWrapper<Dish> dishQueryWrapper = new QueryWrapper<>();
            dishQueryWrapper.in("store_id", storeIds);
            List<Dish> dishes = dishService.list(dishQueryWrapper);
            if (dishes == null || dishes.isEmpty()) {
                // 如果没有菜品，返回空数据
                Map<String, Object> result = new HashMap<>();
                result.put("dates", dates);
                result.put("orderCounts", orderCounts);
                return success(result);
            }
            
            // 提取菜品ID列表
            List<Long> dishIds = dishes.stream().map(Dish::getId).collect(Collectors.toList());
            
            // 查询指定时间范围内的订单
            QueryWrapper<Order> orderQueryWrapper = new QueryWrapper<>();
            orderQueryWrapper.in("dish_id", dishIds);
            orderQueryWrapper.between("create_time", startDate, endDate);
            List<Order> orders = orderService.list(orderQueryWrapper);
            
            // 按日期分组统计
            Map<String, Integer> orderCountMap = new LinkedHashMap<>();
            
            // 生成日期列表
            Calendar tempCalendar = Calendar.getInstance();
            tempCalendar.setTime(startDate);
            
            SimpleDateFormat dateFormat = null;
            if (interval == Calendar.MONTH) {
                dateFormat = new SimpleDateFormat("yyyy-MM");
            } else {
                dateFormat = new SimpleDateFormat("MM-dd");
            }
            
            while (!tempCalendar.getTime().after(endDate)) {
                String dateStr = dateFormat.format(tempCalendar.getTime());
                dates.add(dateStr);
                orderCountMap.put(dateStr, 0);
                
                if (interval == Calendar.MONTH) {
                    tempCalendar.add(Calendar.MONTH, 1);
                } else {
                    tempCalendar.add(Calendar.DAY_OF_MONTH, 1);
                }
            }
            
            // 统计订单数
            for (Order order : orders) {
                String dateStr = dateFormat.format(order.getCreateTime());
                orderCountMap.put(dateStr, orderCountMap.getOrDefault(dateStr, 0) + 1);
            }
            
            // 转换为列表
            for (String date : dates) {
                orderCounts.add(orderCountMap.getOrDefault(date, 0));
            }
            
            Map<String, Object> result = new HashMap<>();
            result.put("dates", dates);
            result.put("orderCounts", orderCounts);
            
            return success(result);
        } catch (Exception e) {
            e.printStackTrace();
            // 如果出现异常，返回空数据
            Map<String, Object> result = new HashMap<>();
            result.put("dates", new ArrayList<>());
            result.put("orderCounts", new ArrayList<>());
            return success(result);
        }
    }

    /**
     * 获取营业额趋势数据
     *
     * @param merchantId 商家ID
     * @param timeRange 时间范围（week/month/quarter/year/7days/30days/90days/180days）
     * @return 营业额趋势数据
     */
    @GetMapping("/revenue/trend")
    public Result getRevenueTrend(
            @RequestParam Long merchantId,
            @RequestParam(defaultValue = "week") String timeRange) {
        try {
            // 从数据库获取真实营业额趋势数据
            List<String> dates = new ArrayList<>();
            List<BigDecimal> revenues = new ArrayList<>();
            
            // 计算时间范围
            Calendar calendar = Calendar.getInstance();
            Date endDate = calendar.getTime();
            Date startDate = null;
            int interval = Calendar.DAY_OF_MONTH; // 时间间隔，默认为1天
            
            // 处理前端传递的时间范围值
            switch (timeRange) {
                case "week":
                case "7days":
                    // 最近7天
                    calendar.add(Calendar.DAY_OF_MONTH, -6);
                    startDate = calendar.getTime();
                    break;
                case "month":
                case "30days":
                    // 最近30天
                    calendar.add(Calendar.DAY_OF_MONTH, -29);
                    startDate = calendar.getTime();
                    break;
                case "quarter":
                case "90days":
                    // 最近90天
                    calendar.add(Calendar.DAY_OF_MONTH, -89);
                    startDate = calendar.getTime();
                    break;
                case "180days":
                    // 最近180天
                    calendar.add(Calendar.DAY_OF_MONTH, -179);
                    startDate = calendar.getTime();
                    break;
                case "year":
                    // 最近12个月
                    calendar.add(Calendar.MONTH, -11);
                    startDate = calendar.getTime();
                    interval = Calendar.MONTH;
                    break;
                default:
                    // 默认最近7天
                    calendar.add(Calendar.DAY_OF_MONTH, -6);
                    startDate = calendar.getTime();
            }
            
            // 获取商家所有店面
            List<Store> stores = storeService.getByMerchantId(merchantId);
            if (stores == null || stores.isEmpty()) {
                // 如果没有店面，返回空数据
                Map<String, Object> result = new HashMap<>();
                result.put("dates", dates);
                result.put("revenues", revenues);
                return success(result);
            }
            
            // 构建店面ID列表
            List<Long> storeIds = stores.stream().map(Store::getId).toList();
            
            // 查询该商家所有店面的所有菜品
            QueryWrapper<Dish> dishQueryWrapper = new QueryWrapper<>();
            dishQueryWrapper.in("store_id", storeIds);
            List<Dish> dishes = dishService.list(dishQueryWrapper);
            if (dishes == null || dishes.isEmpty()) {
                // 如果没有菜品，返回空数据
                Map<String, Object> result = new HashMap<>();
                result.put("dates", dates);
                result.put("revenues", revenues);
                return success(result);
            }
            
            // 提取菜品ID列表
            List<Long> dishIds = dishes.stream().map(Dish::getId).collect(Collectors.toList());
            
            // 查询指定时间范围内的订单
            QueryWrapper<Order> orderQueryWrapper = new QueryWrapper<>();
            orderQueryWrapper.in("dish_id", dishIds);
            orderQueryWrapper.between("create_time", startDate, endDate);
            List<Order> orders = orderService.list(orderQueryWrapper);
            
            // 按日期分组统计营业额
            Map<String, BigDecimal> revenueMap = new LinkedHashMap<>();
            
            // 生成日期列表
            Calendar tempCalendar = Calendar.getInstance();
            tempCalendar.setTime(startDate);
            
            SimpleDateFormat dateFormat = null;
            if (interval == Calendar.MONTH) {
                dateFormat = new SimpleDateFormat("yyyy-MM");
            } else {
                dateFormat = new SimpleDateFormat("MM-dd");
            }
            
            while (!tempCalendar.getTime().after(endDate)) {
                String dateStr = dateFormat.format(tempCalendar.getTime());
                dates.add(dateStr);
                revenueMap.put(dateStr, BigDecimal.ZERO);
                
                if (interval == Calendar.MONTH) {
                    tempCalendar.add(Calendar.MONTH, 1);
                } else {
                    tempCalendar.add(Calendar.DAY_OF_MONTH, 1);
                }
            }
            
            // 统计每天的营业额
            for (Order order : orders) {
                String dateStr = dateFormat.format(order.getCreateTime());
                BigDecimal orderAmount = order.getTotalPrice() != null ? order.getTotalPrice() : BigDecimal.ZERO;
                revenueMap.put(dateStr, revenueMap.get(dateStr).add(orderAmount));
            }
            
            // 转换为列表
            for (String date : dates) {
                revenues.add(revenueMap.get(date).setScale(2, BigDecimal.ROUND_HALF_UP));
            }
            
            Map<String, Object> result = new HashMap<>();
            result.put("dates", dates);
            result.put("revenues", revenues);
            
            return success(result);
        } catch (Exception e) {
            e.printStackTrace();
            // 如果出现异常，返回空数据
            Map<String, Object> result = new HashMap<>();
            result.put("dates", new ArrayList<>());
            result.put("revenues", new ArrayList<>());
            return success(result);
        }
    }

    /**
     * 获取菜品销量排名数据
     *
     * @param merchantId 商家ID
     * @param limit 限制数量
     * @return 菜品销量排名数据
     */
    @GetMapping("/dish/sales/ranking")
    public Result getDishSalesRanking(
            @RequestParam Long merchantId,
            @RequestParam(defaultValue = "10") Integer limit) {
        // 获取商家所有店面
        List<Store> stores = storeService.getByMerchantId(merchantId);
        if (stores.isEmpty()) {
            return success(new HashMap<String, Object>() {
                {
                    put("dishNames", new ArrayList<>());
                    put("sales", new ArrayList<>());
                }
            });
        }

        // 构建店面ID列表
        List<Long> storeIds = stores.stream().map(Store::getId).toList();

        // 获取商家所有菜品，按创建时间降序排序（暂时替代销量排序）
        QueryWrapper<Dish> dishQueryWrapper = new QueryWrapper<>();
        dishQueryWrapper.in("store_id", storeIds)
                .orderByDesc("create_time")
                .last("LIMIT " + limit);
        List<Dish> dishes = dishService.list(dishQueryWrapper);

        // 构建返回数据
        List<String> dishNames = dishes.stream().map(Dish::getName).toList();
        // 根据实际业务逻辑从订单中统计销量，这里暂时返回模拟数据
        List<Integer> sales = new ArrayList<>();
        Random random = new Random();
        for (int i = 0; i < dishes.size(); i++) {
            sales.add(random.nextInt(100) + 10); // 生成10-110之间的随机销量
        }

        Map<String, Object> result = new HashMap<>();
        result.put("dishNames", dishNames);
        result.put("sales", sales);

        return success(result);
    }

    /**
     * 获取顾客分布数据
     *
     * @param merchantId 商家ID
     * @return 顾客分布数据
     */
    @GetMapping("/customer/distribution")
    public Result getCustomerDistribution(@RequestParam Long merchantId) {
        // 模拟数据，后续需要根据实际数据库设计进行修改
        List<String> regions = Arrays.asList("朝阳区", "海淀区", "东城区", "西城区", "丰台区", "石景山区", "通州区", "大兴区");
        List<Integer> counts = new ArrayList<>();

        for (int i = 0; i < regions.size(); i++) {
            counts.add(new Random().nextInt(500) + 100);
        }

        Map<String, Object> result = new HashMap<>();
        result.put("regions", regions);
        result.put("counts", counts);

        return success(result);
    }

    /**
     * 获取订单详细数据
     *
     * @param merchantId 商家ID
     * @param timeRange 时间范围
     * @return 订单详细数据
     */
    @GetMapping("/order/detail")
    public Result getOrderDetail(@RequestParam Long merchantId, @RequestParam String timeRange) {
        // 模拟数据，后续需要根据实际数据库设计进行修改
        List<Map<String, Object>> orderData = new ArrayList<>();
        
        // 生成模拟数据
        Calendar calendar = Calendar.getInstance();
        int days = 30;
        if ("7days".equals(timeRange)) {
            days = 7;
        } else if ("90days".equals(timeRange)) {
            days = 90;
        } else if ("180days".equals(timeRange)) {
            days = 180;
        } else if ("365days".equals(timeRange)) {
            days = 365;
        }
        
        for (int i = days - 1; i >= 0; i--) {
            calendar.add(Calendar.DAY_OF_MONTH, -1);
            Date date = calendar.getTime();
            String dateStr = new java.text.SimpleDateFormat("yyyy-MM-dd").format(date);
            
            Map<String, Object> order = new HashMap<>();
            order.put("date", dateStr);
            order.put("orders", new Random().nextInt(50) + 20);
            order.put("revenue", new BigDecimal(new Random().nextInt(5000) + 2000).setScale(2, BigDecimal.ROUND_HALF_UP));
            order.put("avgOrderValue", new BigDecimal(new Random().nextInt(200) + 50).setScale(2, BigDecimal.ROUND_HALF_UP));
            
            orderData.add(order);
        }
        
        return success(orderData);
    }

    /**
     * 获取菜品详细数据
     *
     * @param merchantId 商家ID
     * @param timeRange 时间范围
     * @return 菜品详细数据
     */
    @GetMapping("/dish/detail")
    public Result getDishDetail(@RequestParam Long merchantId, @RequestParam String timeRange) {
        // 获取商家所有店面
        List<Store> stores = storeService.getByMerchantId(merchantId);
        if (stores.isEmpty()) {
            return success(new ArrayList<>());
        }

        // 构建店面ID列表
        List<Long> storeIds = stores.stream().map(Store::getId).toList();

        // 获取商家所有菜品
        QueryWrapper<Dish> dishQueryWrapper = new QueryWrapper<>();
        dishQueryWrapper.in("store_id", storeIds);
        List<Dish> dishes = dishService.list(dishQueryWrapper);
        
        // 转换为前端需要的格式
        List<Map<String, Object>> dishData = new ArrayList<>();
        Random random = new Random();
        for (Dish dish : dishes) {
            Map<String, Object> dishMap = new HashMap<>();
            // 使用正确的字段名getName()，而不是getDishName()
            dishMap.put("dishName", dish.getName());
            // 生成随机销量，因为Dish实体类没有sales字段
            int sales = random.nextInt(100) + 10;
            dishMap.put("sales", sales);
            // 计算收入，处理可能的null值
            BigDecimal price = dish.getPrice() != null ? dish.getPrice() : BigDecimal.ZERO;
            dishMap.put("revenue", price.multiply(new BigDecimal(sales)));
            // 生成随机评分，因为Dish实体类没有rating字段
            dishMap.put("rating", new BigDecimal(random.nextDouble() * 2 + 3).setScale(1, BigDecimal.ROUND_HALF_UP));
            dishData.add(dishMap);
        }
        
        return success(dishData);
    }

    /**
     * 获取店面详细数据
     *
     * @param merchantId 商家ID
     * @param timeRange 时间范围
     * @return 店面详细数据
     */
    @GetMapping("/store/detail")
    public Result getStoreDetail(@RequestParam Long merchantId, @RequestParam String timeRange) {
        // 获取商家所有店面
        List<Store> stores = storeService.getByMerchantId(merchantId);
        
        // 转换为前端需要的格式
        List<Map<String, Object>> storeData = new ArrayList<>();
        for (Store store : stores) {
            Map<String, Object> storeMap = new HashMap<>();
            storeMap.put("storeName", store.getName());
            storeMap.put("orders", new Random().nextInt(500) + 200);
            storeMap.put("revenue", new BigDecimal(new Random().nextInt(50000) + 20000).setScale(2, BigDecimal.ROUND_HALF_UP));
            storeMap.put("customers", new Random().nextInt(300) + 100);
            storeData.add(storeMap);
        }
        
        return success(storeData);
    }
}
