package com.shiguangjianke.smartfoodsystem.controller;

import com.shiguangjianke.smartfoodsystem.dto.LoginDto;
import com.shiguangjianke.smartfoodsystem.entity.User;
import com.shiguangjianke.smartfoodsystem.service.UserService;
import com.shiguangjianke.smartfoodsystem.service.impl.UserServiceImpl;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

/**
 * 认证Controller
 *
 * @author shiguangjianke
 * @since 2024-07-28
 */
@RestController
@RequestMapping("/api/auth")
public class AuthController extends BaseController {
    
    @Autowired
    private UserService userService;
    
    /**
     * 微信登录
     *
     * @param openid 微信openid
     * @return 登录结果
     */
    @PostMapping("/wechat/login")
    public Result wechatLogin(@RequestParam String openid) {
        // 根据openid查询用户
        User user = userService.getByOpenid(openid);
        if (user == null) {
            // 用户不存在，创建新用户
            user = new User();
            user.setOpenid(openid);
            user.setUserType(1); // 学生用户
            userService.save(user);
        }
        return success(user);
    }
    
    /**
     * 用户登录
     *
     * @param loginDto 登录信息
     * @return 登录结果
     */
    @PostMapping("/user/login")
    public Result userLogin(@RequestBody LoginDto loginDto) {
        // 调用普通用户登录方法
        User user = ((UserServiceImpl) userService).userLogin(loginDto.getUsername(), loginDto.getPassword());
        if (user == null) {
            return error(401, "用户名或密码错误");
        }
        return success(user);
    }
    
    /**
     * 商家登录
     *
     * @param loginDto 登录信息
     * @return 登录结果
     */
    @PostMapping("/merchant/login")
    public Result merchantLogin(@RequestBody LoginDto loginDto) {
        User user = userService.merchantLogin(loginDto.getUsername(), loginDto.getPassword());
        if (user == null) {
            return error(401, "用户名或密码错误");
        }
        return success(user);
    }
    
    /**
     * 管理员登录
     *
     * @param loginDto 登录信息
     * @return 登录结果
     */
    @PostMapping("/admin/login")
    public Result adminLogin(@RequestBody LoginDto loginDto) {
        User user = userService.adminLogin(loginDto.getUsername(), loginDto.getPassword());
        if (user == null) {
            return error(401, "用户名或密码错误");
        }
        return success(user);
    }
    
    /**
     * 用户注册
     *
     * @param user 用户信息
     * @return 注册结果
     */
    @PostMapping("/user/register")
    public Result userRegister(@RequestBody User user) {
        user.setUserType(1); // 普通用户
        // 对密码进行MD5加密
        String encryptedPassword = org.springframework.util.DigestUtils.md5DigestAsHex(user.getPassword().getBytes());
        user.setPassword(encryptedPassword);
        userService.save(user);
        return success("注册成功");
    }
    
    /**
     * 商家注册
     *
     * @param user 商家信息
     * @return 注册结果
     */
    @PostMapping("/merchant/register")
    public Result merchantRegister(@RequestBody User user) {
        user.setUserType(2); // 商家用户
        user.setMerchantApplyStatus(0); // 申请中
        // 对密码进行MD5加密
        String encryptedPassword = org.springframework.util.DigestUtils.md5DigestAsHex(user.getPassword().getBytes());
        user.setPassword(encryptedPassword);
        userService.save(user);
        return success("注册成功，等待审核");
    }
    
    /**
     * 更新商家信息
     *
     * @param user 商家信息
     * @return 更新结果
     */
    @PutMapping("/merchant/update")
    public Result updateMerchant(@RequestBody User user) {
        userService.updateById(user);
        return success("更新成功");
    }
    
    /**
     * 更新用户信息
     *
     * @param user 用户信息
     * @return 更新结果
     */
    @PutMapping("/update")
    public Result updateUser(@RequestBody User user) {
        userService.updateById(user);
        return success(user);
    }
}