package com.shiguangjianke.smartfoodsystem.controller;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.shiguangjianke.smartfoodsystem.entity.AdminAction;
import com.shiguangjianke.smartfoodsystem.service.AdminActionService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 管理端系统管理Controller
 *
 * @author shiguangjianke
 * @since 2024-07-28
 */
@RestController
@RequestMapping("/api/admin/system")
public class AdminSystemController extends BaseController {

    @Autowired
    private AdminActionService adminActionService;

    /**
     * 获取系统设置
     *
     * @return 系统设置
     */
    @GetMapping("/settings")
    public Result getSystemSettings() {
        // 示例系统设置数据，实际项目中应该从数据库或配置文件读取
        Map<String, Object> settings = new HashMap<>();
        settings.put("platformName", "食光荐客");
        settings.put("platformDescription", "美食推荐平台");
        settings.put("contactEmail", "contact@shiguangjianke.com");
        settings.put("contactPhone", "400-123-4567");
        settings.put("maxMerchantStores", 5);
        settings.put("evaluationReviewEnabled", true);
        settings.put("feedbackReviewEnabled", true);

        return success(settings);
    }

    /**
     * 更新系统设置
     *
     * @param settings 系统设置
     * @return 更新结果
     */
    @PutMapping("/settings")
    public Result updateSystemSettings(@RequestBody Map<String, Object> settings) {
        // 实际项目中应该将设置保存到数据库或配置文件
        return success("系统设置更新成功");
    }

    /**
     * 获取操作日志列表
     *
     * @param keyword 搜索关键词
     * @param actionType 操作类型
     * @param page 页码
     * @param pageSize 每页数量
     * @return 操作日志列表
     */
    @GetMapping("/logs")
    public Result getOperationLogs(
            @RequestParam(required = false) String keyword,
            @RequestParam(required = false) Integer actionType,
            @RequestParam(defaultValue = "1") Integer page,
            @RequestParam(defaultValue = "10") Integer pageSize) {
        QueryWrapper<AdminAction> queryWrapper = new QueryWrapper<>();

        // 搜索关键词
        if (keyword != null && !keyword.isEmpty()) {
            queryWrapper.like("content", keyword);
        }

        // 操作类型
        if (actionType != null) {
            queryWrapper.eq("action_type", actionType);
        }

        // 按创建时间降序
        queryWrapper.orderByDesc("create_time");

        // 分页查询
        Page<AdminAction> logPage = new Page<>(page, pageSize);
        Page<AdminAction> result = adminActionService.page(logPage, queryWrapper);

        // 转换为前端期望的格式
        Page<Map<String, Object>> formattedResult = new Page<>();
        formattedResult.setTotal(result.getTotal());
        formattedResult.setCurrent(result.getCurrent());
        formattedResult.setSize(result.getSize());
        formattedResult.setPages(result.getPages());
        
        List<Map<String, Object>> formattedRecords = new ArrayList<>();
        for (AdminAction adminAction : result.getRecords()) {
            Map<String, Object> formattedLog = new HashMap<>();
            formattedLog.put("id", adminAction.getId());
            formattedLog.put("operator", "管理员" + adminAction.getAdminId()); // 模拟操作人
            formattedLog.put("operationType", getOperationTypeString(adminAction.getActionType())); // 转换为字符串类型
            formattedLog.put("operationContent", adminAction.getContent());
            formattedLog.put("ipAddress", "127.0.0.1"); // 模拟IP地址
            formattedLog.put("operationTime", adminAction.getCreateTime());
            formattedRecords.add(formattedLog);
        }
        formattedResult.setRecords(formattedRecords);

        return success(formattedResult);
    }
    
    /**
     * 将操作类型转换为字符串
     */
    private String getOperationTypeString(Integer actionType) {
        switch (actionType) {
            case 1: return "audit";
            case 2: return "edit";
            case 3: return "add";
            case 4: return "delete";
            default: return "login";
        }
    }

    /**
     * 获取角色列表
     *
     * @return 角色列表
     */
    @GetMapping("/roles")
    public Result getRoles() {
        // 示例角色数据，实际项目中应该从数据库查询
        List<Map<String, Object>> roles = List.of(
                createRole(1L, "超级管理员", "拥有所有权限"),
                createRole(2L, "管理员", "拥有大部分管理权限"),
                createRole(3L, "审核员", "仅拥有审核权限")
        );

        return success(roles);
    }

    /**
     * 获取权限列表
     *
     * @return 权限列表
     */
    @GetMapping("/permissions")
    public Result getPermissions() {
        // 示例权限数据，实际项目中应该从数据库查询
        List<Map<String, Object>> permissions = List.of(
                createPermission(1L, "系统设置", "/api/admin/system/settings"),
                createPermission(2L, "日志管理", "/api/admin/system/logs"),
                createPermission(3L, "角色管理", "/api/admin/system/roles"),
                createPermission(4L, "商家管理", "/api/user/merchant/list"),
                createPermission(5L, "店面管理", "/api/store/admin/list"),
                createPermission(6L, "评价管理", "/api/evaluation/admin/list"),
                createPermission(7L, "反馈管理", "/api/feedback/admin/list")
        );

        return success(permissions);
    }

    /**
     * 获取角色权限
     *
     * @param roleId 角色ID
     * @return 角色权限
     */
    @GetMapping("/roles/{roleId}/permissions")
    public Result getRolePermissions(@PathVariable Long roleId) {
        // 示例角色权限数据，实际项目中应该从数据库查询
        List<Long> permissionIds = List.of(1L, 2L, 3L, 4L, 5L, 6L, 7L);
        return success(permissionIds);
    }

    /**
     * 更新角色权限
     *
     * @param roleId 角色ID
     * @param permissionIds 权限ID列表
     * @return 更新结果
     */
    @PutMapping("/roles/{roleId}/permissions")
    public Result updateRolePermissions(
            @PathVariable Long roleId,
            @RequestBody List<Long> permissionIds) {
        // 实际项目中应该将角色权限关联保存到数据库
        return success("角色权限更新成功");
    }

    /**
     * 创建角色
     */
    private Map<String, Object> createRole(Long id, String name, String description) {
        Map<String, Object> role = new HashMap<>();
        role.put("id", id);
        role.put("name", name);
        role.put("description", description);
        return role;
    }

    /**
     * 创建权限
     */
    private Map<String, Object> createPermission(Long id, String name, String path) {
        Map<String, Object> permission = new HashMap<>();
        permission.put("id", id);
        permission.put("name", name);
        permission.put("path", path);
        return permission;
    }
}