package com.shiguangjianke.smartfoodsystem.controller;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.shiguangjianke.smartfoodsystem.entity.Order;
import com.shiguangjianke.smartfoodsystem.entity.Store;
import com.shiguangjianke.smartfoodsystem.entity.User;
import com.shiguangjianke.smartfoodsystem.service.OrderService;
import com.shiguangjianke.smartfoodsystem.service.StoreService;
import com.shiguangjianke.smartfoodsystem.service.UserService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.math.BigDecimal;
import java.util.*;

/**
 * 管理端数据报表Controller
 *
 * @author shiguangjianke
 * @since 2024-07-28
 */
@RestController
@RequestMapping("/api/admin/data-report")
public class AdminDataReportController extends BaseController {

    @Autowired
    private UserService userService;

    @Autowired
    private StoreService storeService;

    @Autowired
    private OrderService orderService;

    /**
     * 获取数据报表基本统计
     *
     * @return 基本统计数据
     */
    @GetMapping("/basic")
    public Result getBasicReport() {
        Map<String, Object> result = new HashMap<>();

        // 总用户数
        QueryWrapper<User> userQuery = new QueryWrapper<>();
        userQuery.eq("user_type", 1);
        long totalUsers = userService.count(userQuery);
        result.put("totalUsers", totalUsers);

        // 总商家数
        QueryWrapper<User> merchantQuery = new QueryWrapper<>();
        merchantQuery.eq("user_type", 2);
        long totalMerchants = userService.count(merchantQuery);
        result.put("totalMerchants", totalMerchants);

        // 总评价数（模拟数据）
        long totalEvaluations = 1250;
        result.put("totalEvaluations", totalEvaluations);

        // 平均评分（模拟数据）
        double avgRating = 4.2;
        result.put("avgRating", avgRating);

        return success(result);
    }

    /**
     * 获取用户增长趋势
     *
     * @param timeRange 时间范围（week/month/quarter/year）
     * @return 用户增长趋势数据
     */
    @GetMapping("/user-growth")
    public Result getUserGrowthTrend(@RequestParam(defaultValue = "month") String timeRange) {
        // 示例数据，实际项目中应该从数据库查询
        List<Map<String, Object>> result = new ArrayList<>();
        List<String> dates = Arrays.asList("1日", "2日", "3日", "4日", "5日", "6日", "7日");
        List<Integer> values = Arrays.asList(150, 120, 180, 160, 220, 200, 280);

        for (int i = 0; i < dates.size(); i++) {
            Map<String, Object> item = new HashMap<>();
            item.put("date", dates.get(i));
            item.put("value", values.get(i));
            result.add(item);
        }

        return success(result);
    }

    /**
     * 获取商家地区分布
     *
     * @return 商家地区分布数据
     */
    @GetMapping("/merchant-distribution")
    public Result getMerchantDistribution() {
        // 示例数据，实际项目中应该从数据库查询
        List<Map<String, Object>> result = new ArrayList<>();
        String[] regions = {"北京", "上海", "广州", "深圳", "杭州", "成都"};
        int[] counts = {120, 150, 90, 80, 70, 60};

        for (int i = 0; i < regions.length; i++) {
            Map<String, Object> item = new HashMap<>();
            item.put("name", regions[i]);
            item.put("value", counts[i]);
            result.add(item);
        }

        return success(result);
    }

    /**
     * 获取评价评分分布
     *
     * @return 评价评分分布数据
     */
    @GetMapping("/evaluation-score")
    public Result getEvaluationScoreDistribution() {
        // 示例数据，实际项目中应该从数据库查询
        List<Map<String, Object>> result = new ArrayList<>();
        String[] scores = {"1分", "2分", "3分", "4分", "5分"};
        int[] counts = {10, 20, 50, 120, 300};

        for (int i = 0; i < scores.length; i++) {
            Map<String, Object> item = new HashMap<>();
            item.put("name", scores[i]);
            item.put("value", counts[i]);
            item.put("type", "评价分布");
            result.add(item);
        }

        return success(result);
    }

    /**
     * 获取详细数据列表
     *
     * @param page 页码
     * @param pageSize 每页数量
     * @param type 报表类型（daily/weekly/monthly/yearly）
     * @param dateRange 日期范围
     * @return 详细数据列表
     */
    @GetMapping("/detail-list")
    public Result getDetailList(
            @RequestParam(defaultValue = "1") Integer page,
            @RequestParam(defaultValue = "10") Integer pageSize,
            @RequestParam String type,
            @RequestParam(required = false) String dateRange) {
        // 示例数据，实际项目中应该根据type和dateRange从数据库查询数据
        List<Map<String, Object>> data = new ArrayList<>();
        
        // 生成示例数据
        for (int i = 0; i < pageSize; i++) {
            Map<String, Object> item = new HashMap<>();
            // 根据不同的报表类型生成不同的日期格式
            String date;
            if ("daily".equals(type)) {
                date = "2024-07-" + String.format("%02d", (page - 1) * pageSize + i + 1);
            } else if ("weekly".equals(type)) {
                date = "2024年第" + ((page - 1) * pageSize + i + 1) + "周";
            } else if ("monthly".equals(type)) {
                date = "2024-" + String.format("%02d", (page - 1) * pageSize + i + 1);
            } else {
                date = "202" + ((page - 1) * pageSize + i + 1) + "年";
            }
            
            item.put("date", date);
            item.put("newUsers", (int)(Math.random() * 200) + 50);
            item.put("newMerchants", (int)(Math.random() * 50) + 10);
            item.put("newEvaluations", (int)(Math.random() * 300) + 100);
            item.put("avgRating", Math.round((Math.random() * 1 + 4) * 10) / 10.0);
            item.put("activeUsers", (int)(Math.random() * 500) + 200);
            item.put("activeMerchants", (int)(Math.random() * 100) + 50);
            data.add(item);
        }

        Map<String, Object> result = new HashMap<>();
        result.put("list", data);
        result.put("total", 100); // 示例总数据量
        result.put("page", page);
        result.put("pageSize", pageSize);

        return success(result);
    }
}