package com.shiguangjianke.smartfoodsystem.aspect;

import com.shiguangjianke.smartfoodsystem.entity.AdminAction;
import com.shiguangjianke.smartfoodsystem.service.AdminActionService;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.AfterReturning;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.util.Date;
import java.util.Map;

/**
 * 管理员操作日志AOP切面
 * 用于捕获管理员的所有操作并记录到数据库
 */
@Aspect
@Component
public class AdminLogAspect {

    @Autowired
    private AdminActionService adminActionService;

    /**
     * 定义切入点：拦截所有管理员相关的Controller方法
     */
    @Pointcut("execution(* com.shiguangjianke.smartfoodsystem.controller.*Controller.*(..))")
    public void adminOperationPointcut() {
    }

    /**
     * 方法执行后记录日志
     * @param joinPoint 连接点
     * @param result 方法返回结果
     */
    @AfterReturning(pointcut = "adminOperationPointcut()", returning = "result")
    public void recordAdminOperation(JoinPoint joinPoint, Object result) {
        // 获取当前请求
        ServletRequestAttributes attributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
        if (attributes == null) {
            return;
        }
        HttpServletRequest request = attributes.getRequest();

        // 从请求中获取管理员ID（这里假设从请求参数或header中获取，实际项目中应根据认证机制调整）
        Long adminId = getAdminIdFromRequest(request, joinPoint);
        if (adminId == null) {
            // 如果无法获取管理员ID，暂时跳过日志记录
            return;
        }

        // 构建操作日志
        AdminAction adminAction = new AdminAction();
        adminAction.setAdminId(adminId);
        adminAction.setActionType(getActionType(request.getMethod(), joinPoint.getSignature().getName()));
        adminAction.setTargetId(getTargetIdFromRequest(request, joinPoint));
        adminAction.setContent(buildContent(request, joinPoint));
        adminAction.setCreateTime(new Date());

        // 保存操作日志
        adminActionService.recordAdminAction(adminAction);
    }

    /**
     * 从请求中获取管理员ID
     * @param request 请求对象
     * @param joinPoint 连接点
     * @return 管理员ID
     */
    private Long getAdminIdFromRequest(HttpServletRequest request, JoinPoint joinPoint) {
        // 从请求头中获取用户ID
        String userIdStr = request.getHeader("X-User-Id");
        if (userIdStr != null && !userIdStr.isEmpty()) {
            try {
                // 从请求头中获取用户类型
                String userTypeStr = request.getHeader("X-User-Type");
                if (userTypeStr != null && !userTypeStr.isEmpty()) {
                    Integer userType = Integer.parseInt(userTypeStr);
                    // 只有管理员用户（userType=3）才记录操作日志
                    if (userType == 3) {
                        return Long.parseLong(userIdStr);
                    }
                }
            } catch (NumberFormatException e) {
                // 忽略解析错误
            }
        }

        // 从请求参数中获取管理员ID
        userIdStr = request.getParameter("adminId");
        if (userIdStr != null && !userIdStr.isEmpty()) {
            try {
                return Long.parseLong(userIdStr);
            } catch (NumberFormatException e) {
                // 忽略解析错误
            }
        }

        // 从请求体中获取管理员ID（这里简化处理，实际项目中应根据具体情况调整）
        // 注意：这里无法直接获取请求体，因为请求体只能读取一次

        // 从方法参数中获取管理员ID
        Object[] args = joinPoint.getArgs();
        for (Object arg : args) {
            if (arg instanceof Map) {
                Map<?, ?> map = (Map<?, ?>) arg;
                Object adminIdObj = map.get("adminId");
                if (adminIdObj != null) {
                    try {
                        return Long.parseLong(adminIdObj.toString());
                    } catch (NumberFormatException e) {
                        // 忽略解析错误
                    }
                }
            }
        }

        // 默认返回null，表示无法获取管理员ID
        return null;
    }

    /**
     * 根据请求方法和方法名获取操作类型
     * @param httpMethod HTTP请求方法
     * @param methodName 方法名
     * @return 操作类型
     */
    private Integer getActionType(String httpMethod, String methodName) {
        // 根据方法名和HTTP方法判断操作类型
        if (methodName.contains("audit") || methodName.contains("updateMerchantApplyStatus")) {
            return 1; // 审核商家
        } else if (methodName.contains("feedback") || methodName.contains("updateStatus")) {
            return 2; // 处理反馈
        } else if (methodName.contains("user") && (httpMethod.equals("PUT") || httpMethod.equals("DELETE"))) {
            return 3; // 管理权限
        } else {
            return 0; // 其他操作
        }
    }

    /**
     * 从请求中获取目标ID
     * @param request 请求对象
     * @param joinPoint 连接点
     * @return 目标ID
     */
    private Long getTargetIdFromRequest(HttpServletRequest request, JoinPoint joinPoint) {
        // 从请求路径中获取目标ID
        String requestURI = request.getRequestURI();
        String[] parts = requestURI.split("/");
        for (int i = parts.length - 1; i >= 0; i--) {
            try {
                return Long.parseLong(parts[i]);
            } catch (NumberFormatException e) {
                // 忽略解析错误，继续尝试
            }
        }

        // 从请求参数中获取目标ID
        String targetIdStr = request.getParameter("id");
        if (targetIdStr != null && !targetIdStr.isEmpty()) {
            try {
                return Long.parseLong(targetIdStr);
            } catch (NumberFormatException e) {
                // 忽略解析错误
            }
        }

        // 从方法参数中获取目标ID
        Object[] args = joinPoint.getArgs();
        for (Object arg : args) {
            if (arg instanceof Long) {
                return (Long) arg;
            } else if (arg instanceof String) {
                try {
                    return Long.parseLong((String) arg);
                } catch (NumberFormatException e) {
                    // 忽略解析错误
                }
            }
        }

        return 0L; // 默认值
    }

    /**
     * 构建操作内容
     * @param request 请求对象
     * @param joinPoint 连接点
     * @return 操作内容
     */
    private String buildContent(HttpServletRequest request, JoinPoint joinPoint) {
        StringBuilder content = new StringBuilder();
        
        // 添加基本操作信息
        content.append("操作：").append(request.getMethod()).append(" ").append(request.getRequestURI());
        content.append("，类名：").append(joinPoint.getTarget().getClass().getSimpleName());
        content.append("，方法：").append(joinPoint.getSignature().getName());
        
        // 添加请求参数
        Map<String, String[]> parameterMap = request.getParameterMap();
        if (!parameterMap.isEmpty()) {
            content.append("，请求参数：");
            parameterMap.forEach((key, values) -> {
                content.append(key).append("=");
                if (values.length == 1) {
                    content.append(values[0]);
                } else {
                    content.append("[");
                    for (int i = 0; i < values.length; i++) {
                        if (i > 0) {
                            content.append(",");
                        }
                        content.append(values[i]);
                    }
                    content.append("]");
                }
                content.append(",");
            });
            // 移除最后一个逗号
            content.deleteCharAt(content.length() - 1);
        }
        
        // 添加方法参数
        Object[] args = joinPoint.getArgs();
        if (args != null && args.length > 0) {
            content.append("，方法参数：");
            for (int i = 0; i < args.length; i++) {
                if (i > 0) {
                    content.append(",");
                }
                Object arg = args[i];
                // 跳过HttpServletRequest和HttpServletResponse等对象
                if (arg instanceof HttpServletRequest || arg instanceof javax.servlet.http.HttpServletResponse) {
                    continue;
                }
                // 安全地处理参数，避免toString()方法抛出异常
                if (arg != null) {
                    try {
                        content.append(arg.toString());
                    } catch (Exception e) {
                        content.append("[无法序列化的对象]");
                    }
                } else {
                    content.append("null");
                }
            }
        }
        
        return content.toString();
    }
}