-- "食光荐客"食堂店面推荐平台数据库表结构设计
-- 创建时间：2024-07-28
-- 数据库：MySQL 8.0

-- ----------------------------
-- 1. 用户表 (user)
-- ----------------------------
CREATE TABLE `user` (
  `id` BIGINT(20) NOT NULL AUTO_INCREMENT COMMENT '用户ID',
  `openid` VARCHAR(100) DEFAULT NULL COMMENT '微信OpenID',
  `user_type` INT(1) NOT NULL COMMENT '用户类型（1：学生，2：商家，3：管理方）',
  `username` VARCHAR(50) NOT NULL COMMENT '用户名',
  `password` VARCHAR(100) DEFAULT NULL COMMENT '密码（仅商家和管理方）',
  `avatar` VARCHAR(200) DEFAULT NULL COMMENT '头像URL',
  `phone` VARCHAR(20) DEFAULT NULL COMMENT '手机号',
  `student_id` VARCHAR(20) DEFAULT NULL COMMENT '学号（仅学生）',
  `merchant_apply_status` INT(1) DEFAULT '0' COMMENT '商家入驻申请状态（0：未申请，1：审核中，2：通过，3：拒绝）',
  `create_time` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP COMMENT '创建时间',
  `update_time` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT '更新时间',
  PRIMARY KEY (`id`),
  UNIQUE KEY `uk_openid` (`openid`),
  KEY `idx_user_type` (`user_type`),
  KEY `idx_phone` (`phone`),
  KEY `idx_student_id` (`student_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='用户表';

-- ----------------------------
-- 2. 店面表 (store)
-- ----------------------------
CREATE TABLE `store` (
  `id` BIGINT(20) NOT NULL AUTO_INCREMENT COMMENT '店面ID',
  `name` VARCHAR(100) NOT NULL COMMENT '店面名称',
  `address` VARCHAR(200) NOT NULL COMMENT '店面地址',
  `phone` VARCHAR(20) DEFAULT NULL COMMENT '联系电话',
  `business_hours` VARCHAR(100) DEFAULT NULL COMMENT '营业时间',
  `description` VARCHAR(500) DEFAULT NULL COMMENT '店面描述',
  `logo` VARCHAR(200) DEFAULT NULL COMMENT '店面logo',
  `status` INT(1) NOT NULL DEFAULT '1' COMMENT '店面状态（1：营业，2：休息，3：关闭）',
  `merchant_id` BIGINT(20) DEFAULT NULL COMMENT '商家ID',
  `average_score` DECIMAL(3,2) DEFAULT '0.00' COMMENT '平均评分',
  `evaluation_count` INT(10) DEFAULT '0' COMMENT '评价数量',
  `create_time` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP COMMENT '创建时间',
  `update_time` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT '更新时间',
  PRIMARY KEY (`id`),
  KEY `idx_merchant_id` (`merchant_id`),
  KEY `idx_status` (`status`),
  KEY `idx_average_score` (`average_score`),
  CONSTRAINT `fk_store_merchant` FOREIGN KEY (`merchant_id`) REFERENCES `user` (`id`) ON DELETE SET NULL ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='店面表';

-- ----------------------------
-- 3. 菜品表 (dish)
-- ----------------------------
CREATE TABLE `dish` (
  `id` BIGINT(20) NOT NULL AUTO_INCREMENT COMMENT '菜品ID',
  `store_id` BIGINT(20) DEFAULT NULL COMMENT '店面ID',
  `name` VARCHAR(100) NOT NULL COMMENT '菜品名称',
  `price` DECIMAL(8,2) NOT NULL COMMENT '菜品价格',
  `description` VARCHAR(200) DEFAULT NULL COMMENT '菜品描述',
  `image` VARCHAR(200) DEFAULT NULL COMMENT '菜品图片',
  `is_signature` INT(1) DEFAULT '0' COMMENT '是否招牌菜（0：否，1：是）',
  `create_time` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP COMMENT '创建时间',
  `update_time` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT '更新时间',
  PRIMARY KEY (`id`),
  KEY `idx_store_id` (`store_id`),
  KEY `idx_is_signature` (`is_signature`),
  CONSTRAINT `fk_dish_store` FOREIGN KEY (`store_id`) REFERENCES `store` (`id`) ON DELETE CASCADE ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='菜品表';

-- ----------------------------
-- 4. 评价表 (evaluation)
-- ----------------------------
CREATE TABLE `evaluation` (
  `id` BIGINT(20) NOT NULL AUTO_INCREMENT COMMENT '评价ID',
  `user_id` BIGINT(20) DEFAULT NULL COMMENT '用户ID',
  `store_id` BIGINT(20) DEFAULT NULL COMMENT '店面ID',
  `taste_score` INT(1) NOT NULL COMMENT '口味评分（1-5）',
  `environment_score` INT(1) NOT NULL COMMENT '环境评分（1-5）',
  `service_score` INT(1) NOT NULL COMMENT '服务评分（1-5）',
  `content` TEXT COMMENT '评价内容（5-500字）',
  `images` VARCHAR(1000) DEFAULT NULL COMMENT '图片URL列表（JSON格式）',
  `reply_content` TEXT COMMENT '商家回复内容',
  `reply_time` DATETIME DEFAULT NULL COMMENT '回复时间',
  `create_time` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP COMMENT '创建时间',
  `update_time` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT '更新时间',
  PRIMARY KEY (`id`),
  KEY `idx_user_id` (`user_id`),
  KEY `idx_store_id` (`store_id`),
  KEY `idx_taste_score` (`taste_score`),
  KEY `idx_environment_score` (`environment_score`),
  KEY `idx_service_score` (`service_score`),
  KEY `idx_create_time` (`create_time`),
  CONSTRAINT `fk_evaluation_user` FOREIGN KEY (`user_id`) REFERENCES `user` (`id`) ON DELETE SET NULL ON UPDATE CASCADE,
  CONSTRAINT `fk_evaluation_store` FOREIGN KEY (`store_id`) REFERENCES `store` (`id`) ON DELETE CASCADE ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='评价表';

-- ----------------------------
-- 5. 反馈表 (feedback)
-- ----------------------------
CREATE TABLE `feedback` (
  `id` BIGINT(20) NOT NULL AUTO_INCREMENT COMMENT '反馈ID',
  `user_id` BIGINT(20) DEFAULT NULL COMMENT '用户ID',
  `store_id` BIGINT(20) DEFAULT NULL COMMENT '店面ID',
  `type` INT(1) NOT NULL COMMENT '反馈类型（1：菜品问题，2：服务问题，3：环境问题，4：其他）',
  `content` TEXT NOT NULL COMMENT '反馈内容（5-500字）',
  `images` VARCHAR(1000) DEFAULT NULL COMMENT '图片URL列表（JSON格式）',
  `status` INT(1) DEFAULT '0' COMMENT '处理状态（0：待处理，1：处理中，2：已解决，3：已关闭）',
  `handle_result` TEXT COMMENT '处理结果',
  `handle_admin_id` BIGINT(20) DEFAULT NULL COMMENT '处理管理员ID',
  `handle_time` DATETIME DEFAULT NULL COMMENT '处理时间',
  `create_time` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP COMMENT '创建时间',
  `update_time` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT '更新时间',
  PRIMARY KEY (`id`),
  KEY `idx_user_id` (`user_id`),
  KEY `idx_store_id` (`store_id`),
  KEY `idx_type` (`type`),
  KEY `idx_status` (`status`),
  KEY `idx_handle_admin_id` (`handle_admin_id`),
  KEY `idx_create_time` (`create_time`),
  CONSTRAINT `fk_feedback_user` FOREIGN KEY (`user_id`) REFERENCES `user` (`id`) ON DELETE SET NULL ON UPDATE CASCADE,
  CONSTRAINT `fk_feedback_store` FOREIGN KEY (`store_id`) REFERENCES `store` (`id`) ON DELETE SET NULL ON UPDATE CASCADE,
  CONSTRAINT `fk_feedback_admin` FOREIGN KEY (`handle_admin_id`) REFERENCES `user` (`id`) ON DELETE SET NULL ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='反馈表';

-- ----------------------------
-- 6. 收藏表 (favorite)
-- ----------------------------
CREATE TABLE `favorite` (
  `id` BIGINT(20) NOT NULL AUTO_INCREMENT COMMENT '收藏ID',
  `user_id` BIGINT(20) DEFAULT NULL COMMENT '用户ID',
  `store_id` BIGINT(20) DEFAULT NULL COMMENT '店面ID',
  `create_time` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP COMMENT '创建时间',
  PRIMARY KEY (`id`),
  UNIQUE KEY `uk_user_store` (`user_id`, `store_id`),
  KEY `idx_user_id` (`user_id`),
  KEY `idx_store_id` (`store_id`),
  CONSTRAINT `fk_favorite_user` FOREIGN KEY (`user_id`) REFERENCES `user` (`id`) ON DELETE CASCADE ON UPDATE CASCADE,
  CONSTRAINT `fk_favorite_store` FOREIGN KEY (`store_id`) REFERENCES `store` (`id`) ON DELETE CASCADE ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='收藏表';

-- ----------------------------
-- 7. 管理员操作表 (admin_action)
-- ----------------------------
CREATE TABLE `admin_action` (
  `id` BIGINT(20) NOT NULL AUTO_INCREMENT COMMENT '操作ID',
  `admin_id` BIGINT(20) DEFAULT NULL COMMENT '管理员ID',
  `action_type` INT(1) NOT NULL COMMENT '操作类型（1：审核商家，2：处理反馈，3：管理权限）',
  `target_id` BIGINT(20) NOT NULL COMMENT '目标ID（商家ID/反馈ID/用户ID）',
  `content` TEXT COMMENT '操作内容',
  `create_time` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP COMMENT '创建时间',
  PRIMARY KEY (`id`),
  KEY `idx_admin_id` (`admin_id`),
  KEY `idx_action_type` (`action_type`),
  KEY `idx_target_id` (`target_id`),
  KEY `idx_create_time` (`create_time`),
  CONSTRAINT `fk_admin_action_user` FOREIGN KEY (`admin_id`) REFERENCES `user` (`id`) ON DELETE SET NULL ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='管理员操作表';

-- ----------------------------
-- 8. 初始化数据
-- ----------------------------
-- 初始化管理方账号
INSERT INTO `user` (`user_type`, `username`, `password`, `avatar`, `phone`) 
VALUES (3, 'admin', '$2a$10$7x9XaBvNcOdPeQrStUvWx.YzAbCdEfGhIjKlMnOpQrStUvWxYz', 'https://example.com/avatar/admin.png', '13800138000');

-- 初始化用户类型枚举说明
INSERT INTO `user` (`id`, `user_type`, `username`) VALUES (0, 1, '学生');
INSERT INTO `user` (`id`, `user_type`, `username`) VALUES (0, 2, '商家');
INSERT INTO `user` (`id`, `user_type`, `username`) VALUES (0, 3, '管理方');

-- 初始化反馈类型枚举说明
INSERT INTO `feedback` (`id`, `type`) VALUES (0, 1);
INSERT INTO `feedback` (`id`, `type`) VALUES (0, 2);
INSERT INTO `feedback` (`id`, `type`) VALUES (0, 3);
INSERT INTO `feedback` (`id`, `type`) VALUES (0, 4);

-- 初始化管理员操作类型枚举说明
INSERT INTO `admin_action` (`id`, `action_type`) VALUES (0, 1);
INSERT INTO `admin_action` (`id`, `action_type`) VALUES (0, 2);
INSERT INTO `admin_action` (`id`, `action_type`) VALUES (0, 3);

-- 提交事务
COMMIT;