import axios from 'axios'
import { ElMessage } from 'element-plus'
import { useUserStore } from '../store/modules/user'
import pinia from '../store'

// 创建axios实例
const request = axios.create({
  baseURL: 'http://localhost:8080/api', // 后端API地址，包含/api前缀
  timeout: 10000 // 请求超时时间
})

// 请求拦截器
request.interceptors.request.use(
  config => {
    // 从localStorage获取token
    const token = localStorage.getItem('token')
    if (token) {
      // 设置token到请求头
      config.headers['Authorization'] = `Bearer ${token}`
    }
    
    // 从localStorage获取用户信息
    const userStr = localStorage.getItem('user')
    if (userStr) {
      const user = JSON.parse(userStr)
      // 设置用户ID和用户类型到请求头
      if (user.id) {
        config.headers['X-User-Id'] = user.id
      }
      if (user.userType) {
        config.headers['X-User-Type'] = user.userType
      }
    }
    return config
  },
  error => {
    console.error('请求错误:', error)
    return Promise.reject(error)
  }
)

// 响应拦截器
request.interceptors.response.use(
  response => {
    console.log('响应拦截器收到的response:', response)
    const res = response.data
    
    // 如果返回的是二进制数据，直接返回
    if (response.config.responseType === 'blob' || response.config.responseType === 'arraybuffer') {
      return response
    }
    
    console.log('响应拦截器处理的res:', res)
    // 统一处理响应码
    if (res.code !== 200) {
      ElMessage.error(res.message || '请求失败')
      
      // 处理未登录或token过期
      if (res.code === 401) {
        // 获取当前请求路径
        const currentPath = response.config.url
        // 白名单：不需要登录的公共接口
        const whiteList = ['/store/list', '/store/detail/']
        // 检查是否在白名单中
        const isInWhiteList = whiteList.some(path => currentPath.includes(path))
        
        if (!isInWhiteList) {
          const userStore = useUserStore(pinia)
          userStore.logout()
          // 跳转到登录页
          window.location.href = '/login'
        }
      }
      
      return Promise.reject(new Error(res.message || '请求失败'))
    } else {
      return res.data
    }
  },
  error => {
    console.error('响应错误:', error)
    console.error('错误详情:', error.response)
    let message = '网络错误，请稍后重试'
    
    if (error.response) {
      // 服务器返回错误
      const { status, data } = error.response
      console.error('后端返回数据:', data)
      switch (status) {
        case 400:
          message = data.message || '请求参数错误'
          break
        case 401:
          message = '未登录，请重新登录'
          // 获取当前请求路径
          const currentPath = error.response.config.url
          // 白名单：不需要登录的公共接口
          const whiteList = ['/store/list', '/store/detail/']
          // 检查是否在白名单中
          const isInWhiteList = whiteList.some(path => currentPath.includes(path))
          
          if (!isInWhiteList) {
            const userStore = useUserStore(pinia)
            userStore.logout()
            window.location.href = '/login'
          }
          break
        case 403:
          message = '没有权限访问'
          break
        case 404:
          message = data.message || '请求的资源不存在'
          break
        case 500:
          message = data.message || '服务器内部错误'
          break
        default:
          message = data.message || `请求失败(${status})`
      }
    } else if (error.request) {
      // 请求已发送但没有收到响应
      message = '服务器无响应，请稍后重试'
    } else {
      // 请求配置错误
      message = error.message
    }
    
    ElMessage.error(message)
    return Promise.reject(new Error(message))
  }
)

export default request