import { defineStore } from 'pinia'
import request from '../../utils/request'

export const useUserStore = defineStore('user', {
  state: () => ({
    userInfo: JSON.parse(localStorage.getItem('user')) || null,
    token: localStorage.getItem('token') || '',
    isLogin: !!localStorage.getItem('token') || !!localStorage.getItem('user')
  }),

  getters: {
    user: (state) => state.userInfo,
    isAuthenticated: (state) => state.isLogin
  },

  actions: {
    // 登录
    async login(loginForm, loginType = 'user') {
      try {
        console.log('=== 开始登录流程 ===')
        console.log('登录类型:', loginType)
        console.log('登录表单数据:', loginForm)
        
        // 根据登录类型选择不同的接口
        let url = '/auth/user/login'
        if (loginType === 'admin') {
          url = '/auth/admin/login'
          console.log('管理员登录，使用URL:', url)
        } else if (loginType === 'merchant') {
          url = '/auth/merchant/login'
          console.log('商家登录，使用URL:', url)
        } else {
          console.log('普通用户登录，使用URL:', url)
        }
        
        console.log('准备发送登录请求...')
        const response = await request.post(url, loginForm)
        console.log('登录请求响应:', response)
        
        // 检查响应结构
        if (!response) {
          throw new Error('响应为空')
        }
        
        // 由于响应拦截器处理，成功时直接返回data，失败时抛出异常
        // 所以这里直接处理响应数据
        let user = response || {}
        console.log('处理前的用户对象:', user)
        
        if (user.userType) {
          console.log('使用userType设置角色:', user.userType)
          switch (user.userType) {
            case 3:
              user.role = 'admin'
              console.log('设置角色为admin')
              break
            case 2:
              user.role = 'merchant'
              console.log('设置角色为merchant')
              break
            default:
              user.role = 'user'
              console.log('设置角色为user')
          }
        } else {
          // 如果后端没有返回userType，根据登录类型设置默认角色
          user.role = loginType
          console.log('使用登录类型设置角色:', user.role)
        }
        
        // 确保用户对象有id和nickname属性
        if (!user.id) {
          user.id = Date.now()
          console.log('生成临时id:', user.id)
        }
        if (!user.nickname) {
          user.nickname = loginForm.username || loginForm.userName || '用户'
          console.log('设置昵称:', user.nickname)
        }
        
        // 保存token和用户信息
        // 由于当前后端没有返回token，这里使用空字符串
        this.token = ''
        this.userInfo = user
        this.isLogin = true
        
        console.log('更新store状态:', {
          token: this.token,
          userInfo: this.userInfo,
          isLogin: this.isLogin
        })
        
        // 存储到localStorage
        localStorage.setItem('token', '')
        localStorage.setItem('user', JSON.stringify(user))
        console.log('保存到localStorage:', {
          token: localStorage.getItem('token'),
          user: localStorage.getItem('user')
        })
        
        console.log('=== 登录流程结束 ===')
        return { code: 200, message: '登录成功', data: user }
      } catch (error) {
        console.error('=== 登录流程出错 ===')
        console.error('错误类型:', typeof error)
        console.error('错误对象:', error)
        console.error('错误信息:', error.message)
        if (error.response) {
          console.error('响应错误状态:', error.response.status)
          console.error('响应错误数据:', error.response.data)
        }
        throw error
      }
    },

    // 注册
    async register(registerForm) {
      try {
        const response = await request.post('/auth/register', registerForm)
        return response
      } catch (error) {
        console.error('注册失败:', error)
        throw error
      }
    },

    // 登出
    logout() {
      this.userInfo = null
      this.token = ''
      this.isLogin = false
      
      localStorage.removeItem('token')
      localStorage.removeItem('user')
    },

    // 获取用户信息
    async getUserInfo() {
      try {
        const response = await request.get('/auth/user-info')
        const { data } = response
        
        this.userInfo = data.data
        localStorage.setItem('user', JSON.stringify(data.data))
        
        return data
      } catch (error) {
        console.error('获取用户信息失败:', error)
        // 如果获取失败，可能是token过期，清除登录状态
        this.logout()
        throw error
      }
    },

    // 更新用户信息
    async updateUserInfo(userForm) {
      try {
        const response = await request.put('/auth/update', userForm)
        const { data } = response
        
        this.userInfo = data.data
        localStorage.setItem('user', JSON.stringify(data.data))
        
        return data
      } catch (error) {
        console.error('更新用户信息失败:', error)
        throw error
      }
    }
  }
})